﻿namespace com.blong.googleapi.map;

interface

uses 
  java.util,
  android.app,
  android.content,
  android.graphics.drawable,
  com.google.android.maps;

type
  /// <summary>
  /// Simple itemised overlay implementation, as per:
  /// http://developer.android.com/resources/tutorials/views/hello-mapview.html
  /// This overlay allows us to add annotations, or markers, onto a map, where
  /// Google calls each marker an overlay item. So an ItemizedOverlay is really a
  /// collection of individual overlay items.
  /// </summary>
  AnnotationOverlay = public class(ItemizedOverlay)
  private
    var mOverlays: ArrayList<OverlayItem> := new ArrayList<OverlayItem>;
    var mContext: Context;
  protected
    method createItem(idx: Integer): OverlayItem; override;
    method onTap(idx: Integer): Boolean; override;
  public
    constructor (defaultMarker: Drawable; ctx: Context);
    method addOverlay(overlay: OverlayItem);
    method removeOverlay(overlay: OverlayItem);
    method size: Integer; override;
  end;

implementation

/// <summary>
/// Construct the itemised overlay
/// </summary>
/// <param name="defaultMarker">Default Drawable resource for each OverlayItem</param>
/// <param name="ctx">The context that hosts the underlying MapView</param>
constructor AnnotationOverlay(defaultMarker: Drawable; ctx: Context);
begin
  inherited constructor(boundCenterBottom(defaultMarker));
  mContext := ctx;
end;

/// <summary>
/// Add a new OverlayItem
/// </summary>
/// <param name="overlay">OverlayItem to add to this ItemizedOverlay descendant</param>
method AnnotationOverlay.addOverlay(overlay: OverlayItem);
begin
  mOverlays.add(overlay);
  populate
end;

/// <summary>
/// Remove an existing OverlayItem
/// </summary>
/// <param name="overlay">OverlayItem to remove from this ItemizedOverlay descendant</param>
method AnnotationOverlay.removeOverlay(overlay: OverlayItem);
begin
  var idx := mOverlays.indexOf(overlay);
  if idx <> -1 then
  begin
    mOverlays.remove(idx);
    populate
  end
end;

method AnnotationOverlay.createItem(idx: Integer): OverlayItem;
begin
  exit mOverlays[idx]
end;

method AnnotationOverlay.size: Integer;
begin
  exit mOverlays.size
end;

/// <summary>
/// Do something when someone taps the marker on the map
/// </summary>
/// <param name="idx">The tapped OverlayItem's index in the list of OverlayItems</param>
/// <returns></returns>
method AnnotationOverlay.onTap(idx: Integer): Boolean;
begin
  var item := mOverlays.get(idx);
  var dialog := new AlertDialog.Builder(mContext);
  dialog.Title := item.Title;
  dialog.Message := item.Snippet;
  dialog.show;
  exit true;
end;

end.