#define ORIENTATION_APPROACH_1

using System;
using System.Drawing;

using MonoTouch.Foundation;
using MonoTouch.UIKit;

namespace Pages
{
	public partial class InfoViewController : UIViewController
	{
		NSTimer UpdateBatteryStatusTimer;
		NSTimer ClearMotionLabelTimer;
		private PointF StartCoord;
		
		public InfoViewController () : base ("InfoViewController", null)
		{
		}
		
		public override void DidReceiveMemoryWarning ()
		{
			// Releases the view if it doesn't have a superview.
			base.DidReceiveMemoryWarning ();
			
			// Release any cached data, images, etc that aren't in use.
		}
		
		public override void ViewDidLoad ()
		{
			base.ViewDidLoad ();
			
			// Perform any additional setup after loading the view, typically from a nib.
		}
		
		public override void ViewDidUnload ()
		{
			base.ViewDidUnload ();
			
			// Clear any references to subviews of the main view in order to
			// allow the Garbage Collector to collect them sooner.
			//
			// e.g. myOutlet.Dispose (); myOutlet = null;
			
			ReleaseDesignerOutlets ();
		}
		
		public override void ViewDidAppear (bool animated)
		{
			base.ViewDidAppear (animated);
			deviceLabel.Text = String.Format ("{0}, iOS v{1}", DeviceHardware.VersionString, UIDevice.CurrentDevice.SystemVersion);
			UpdateUIMetrics ();
			//Status bar
			statusBarSwitch.On = !UIApplication.SharedApplication.StatusBarHidden;
			statusBarSwitch.ValueChanged += HandleStatusBarSwitchValueChanged;
			//Proximity sensor
			UIDevice.CurrentDevice.ProximityMonitoringEnabled = true;
			if (UIDevice.CurrentDevice.ProximityMonitoringEnabled)
				NSNotificationCenter.DefaultCenter.AddObserver (UIDevice.ProximityStateDidChangeNotification,
			    	(n) => {
					proximityLabel.Text = 
			        	UIDevice.CurrentDevice.ProximityState ? "Proximity detected" : "Proximity not detected"; });
			else
				proximityLabel.Text = "Proximity sensor not available";
			//Battery status
			UpdateBatteryStatusTimer = NSTimer.CreateRepeatingScheduledTimer (
			    60, ReadBatteryStatus);
			ReadBatteryStatus ();
			//Orientation
#if !ORIENTATION_APPROACH_1
			NSNotificationCenter.DefaultCenter.AddObserver(UIDevice.OrientationDidChangeNotification,
				 OrientationChanged);
#endif
			//Shakes
			BecomeFirstResponder ();
			UIApplication.SharedApplication.ApplicationSupportsShakeToEdit = true;
		}
		
		public override void ViewDidDisappear (bool animated)
		{
			//Status bar
			statusBarSwitch.On = true;
			HandleStatusBarSwitchValueChanged (statusBarSwitch, new EventArgs ());
			//Proximity
			if (UIDevice.CurrentDevice.ProximityMonitoringEnabled)
				UIDevice.CurrentDevice.ProximityMonitoringEnabled = false;
			//Battery status
			UpdateBatteryStatusTimer.Invalidate ();
			ResignFirstResponder ();
			base.ViewDidDisappear (animated);
		}
		
		public override bool ShouldAutorotateToInterfaceOrientation (UIInterfaceOrientation toInterfaceOrientation)
		{
			// Return true for supported orientations
			return true;
		}
		
		public override void MotionEnded (UIEventSubtype motion, UIEvent evt)
		{
			if (motion == UIEventSubtype.MotionShake) {
				interactionLabel.Text = "iPhone was shaken";
				SetTimerToClearMotionLabel ();
			}
		}

		private void UpdateUIMetrics ()
		{
			var scrn = UIScreen.MainScreen;
			//iPhone 4 doubles pixel count, but point count remains same
			resolutionLabel.Text = string.Format ("{0}x{1} points, {2}x{3} pixels",
		        scrn.Bounds.Width, scrn.Bounds.Height,
		        scrn.Bounds.Width * scrn.Scale, scrn.Bounds.Height * scrn.Scale);    
			frameSizeLabel.Text = string.Format ("{0}x{1} points", 
		        scrn.ApplicationFrame.Width, scrn.ApplicationFrame.Height);
		}
		
		private void HandleStatusBarSwitchValueChanged (object sender, EventArgs e)
		{
		    UIApplication.SharedApplication.StatusBarHidden = !statusBarSwitch.On;
		    if ((View != null) && (View.Window != null))
		        View.Window.Frame = UIScreen.MainScreen.ApplicationFrame;
		    
		    //Without this, the nav bar is lazy about moving to the right place
		    //Required a public property to be added to the AppDelegate
		    var AppDel = (AppDelegate)UIApplication.SharedApplication.Delegate;
		    var NavController = AppDel.NavController;
		    NavController.SetNavigationBarHidden(true, false);
		    NavController.SetNavigationBarHidden(false, false);  
		    UpdateUIMetrics();
		}
		
#if ORIENTATION_APPROACH_1
		public override void WillAnimateRotation (UIInterfaceOrientation toInterfaceOrientation, double duration)
		{
			base.WillAnimateRotation (toInterfaceOrientation, duration);
			switch (toInterfaceOrientation) {
			case UIInterfaceOrientation.Portrait:
				interactionLabel.Text = "iPhone is oriented normally";
				break;
			case UIInterfaceOrientation.LandscapeLeft:
				interactionLabel.Text = "iPhone has been rotated right";
				break;
			case UIInterfaceOrientation.PortraitUpsideDown:
				interactionLabel.Text = "iPhone is upside down";
				break;
			case UIInterfaceOrientation.LandscapeRight:
				interactionLabel.Text = "iPhone has been rotated left";
				break;
			}
			SetTimerToClearMotionLabel ();
			UpdateUIMetrics ();
		}
#endif
		
		public override bool CanBecomeFirstResponder {
			get { return true; }
		}

		public override void TouchesBegan (NSSet touches, UIEvent evt)
		{
			var touchArray = touches.ToArray<UITouch> ();
			if (touches.Count > 0) {
				var coord = touchArray [0].LocationInView (touchArray [0].View);
				if (touchArray [0].TapCount < 2)
					StartCoord = coord;
				interactionLabel.Text = string.Format ("{0} ({1},{2})",
            DescribeTouch (touchArray [0]), coord.X, coord.Y);
			}
		}
 
		public override void TouchesMoved (NSSet touches, UIEvent evt)
		{
			var touchArray = touches.ToArray<UITouch> ();
			if (touches.Count > 0) {
				var coord = touchArray [0].LocationInView (touchArray [0].View);
				interactionLabel.Text = string.Format ("{0} ({1},{2})",
            DescribeTouch (touchArray [0]), coord.X, coord.Y);
			}
		}
 
		public override void TouchesEnded (NSSet touches, UIEvent evt)
		{
			var touchArray = touches.ToArray<UITouch> ();
			if (touches.Count > 0) {
				var coord = touchArray [0].LocationInView (touchArray [0].View);
				interactionLabel.Text = string.Format ("{0} ({1},{2})->({3},{4})",
            DescribeTouch (touchArray [0]), StartCoord.X, StartCoord.Y, coord.X, coord.Y);
				SetTimerToClearMotionLabel ();
			}
		}

		private void ReadBatteryStatus()
		{
		    var dev = UIDevice.CurrentDevice;
		    dev.BatteryMonitoringEnabled = true;
		    if (dev.BatteryMonitoringEnabled)
		        try
		        {
		            batteryLabel.Text = string.Format("{0}% - {1}", Math.Round(dev.BatteryLevel * 100), dev.BatteryState);
		        }
		        finally
		        {
		            dev.BatteryMonitoringEnabled = false;
		        }
		    else
		    {
		        batteryLabel.Text = "Battery level monitoring not available";
		        UpdateBatteryStatusTimer.Invalidate();
		    }            
		}
		
#if !ORIENTATION_APPROACH_1
		private void OrientationChanged(NSNotification notification)
		{
			switch (InterfaceOrientation)
			{
				case UIInterfaceOrientation.Portrait:
					interactionLabel.Text = "iPhone is oriented normally";
					break;
				case UIInterfaceOrientation.LandscapeLeft:
					interactionLabel.Text = "iPhone has been rotated right";
					break;
				case UIInterfaceOrientation.PortraitUpsideDown:
					interactionLabel.Text = "iPhone is upside down";
					break;
				case UIInterfaceOrientation.LandscapeRight:
					interactionLabel.Text = "iPhone has been rotated left";
					break;
			}
			SetTimerToClearMotionLabel();
			UpdateUIMetrics();
		}
#endif

		private void SetTimerToClearMotionLabel ()
		{
			if (ClearMotionLabelTimer != null)
				ClearMotionLabelTimer.Invalidate ();
			ClearMotionLabelTimer = NSTimer.CreateScheduledTimer (3,
				() =>
			{
				interactionLabel.Text = "None";
				ClearMotionLabelTimer = null;
			});
		}
		
		private string DescribeTouch (UITouch touch)
		{
			string desc;
			switch (touch.TapCount) {
			case 0:
				desc = "Swipe";
				break;
			case 1:
				desc = "Single tap";
				break;
			case 2:
				desc = "Double tap";
				break;
			default:
				desc = "Multiple tap";
				break;
			}
			switch (touch.Phase) {
			case UITouchPhase.Began:
				desc += " started";
				break;
			case UITouchPhase.Moved:
				desc += " moved";
				break;
			case UITouchPhase.Stationary:
				desc += " hasn't moved";
				break;
			case UITouchPhase.Ended:
				desc += " ended";
				break;
			case UITouchPhase.Cancelled:
				desc += " cancelled";
				break;
			}
			return desc;
		}
 
	}
}

